/* 

	USB Data Logger BootLoader, by Mauro Grassi, Dec. 2010.

*/

#include "p18f27j53.h"
#include "boot.h"
#include "usb_device.h"
#include "usb.h"
#include "usb_function_generic.h"
#include "usb_config.h"
#include "flash.h"
#include "delay.h"
#include "led.h"
#include "uart.h"
#include "adc.h"

void mainApp(void);
void WriteWordFlash(unsigned long startaddr, unsigned int data);

#pragma udata bootdata

USB_PACKET				OUTPacket;				// User application buffer for receiving and holding OUT packets sent from the host
USB_PACKET 				INPacket;				// User application buffer for sending IN packets to the host
USB_HANDLE 				USBGenericOutHandle;
USB_HANDLE 				USBGenericInHandle;
unsigned long			flashAddress;
unsigned char           bootReset;
unsigned char			bootFunction;

#if (USE_BOOT_UART)
const rom unsigned char messageBoot[]="Entering BootLoader.\r\n";
#endif

void _boothook(void);

#pragma code _entry_scn=0x000000
void _entry (void)
{
	_asm goto _startup _endasm
}

/* Constant Entry Point For Boot, aligned to erase page size (1024 bytes=0x400) */

#pragma code bootstart=0x1DC00

void _startup (void)
{
	bootFunction=0;
 	_asm goto _boothook _endasm
}

#pragma code boothook=0x1DC20

void _boothook(void)
{
 _asm
    // Initialize the stack pointer
    lfsr 1, _stack
    lfsr 2, _stack

    clrf TBLPTRU, 0 // 1st silicon doesn't do this on POR
_endasm
 	
loop:
  // If user defined __init is not found, the one in clib.lib will be used
  //__init ();
  main();
  mainApp(); 			/* call the usercode */
  goto loop;
}
 
#pragma code bootcode

void InitializeSystem(void)
{
	unsigned int pll_startup_counter;
	
	INTCON=0;
	
	pll_startup_counter=600;
    OSCTUNEbits.PLLEN = 1;  //Enable the PLL and wait 2+ms until the PLL locks before enabling USB module
    while(pll_startup_counter--);
    #if defined(USE_USB_BUS_SENSE_IO)
    	tris_usb_bus_sense = INPUT_PIN;
    #endif
    #if defined(USE_SELF_POWER_SENSE_IO)
    tris_self_power = INPUT_PIN;	
    #endif
   	
    ANCON1=0x0F;
	ANCON0=0xFF;
    
    TRISA=0xFF;
    TRISB=0xFF;
    TRISC=0xFF;
    
    RPOR0=0;
    RPOR1=0;
    RPOR2=0;
    RPOR3=0;
    RPOR4=0;
    RPOR5=0;
    RPOR6=0;
    RPOR7=0;
    RPOR8=0;
    RPOR9=0;
    RPOR10=0;
    RPOR11=0;
    RPOR12=0;
    RPOR13=0;
    RPOR17=0;
    RPOR18=0;
    
	USBGenericOutHandle = 0;	
	USBGenericInHandle = 0;		

	#if(USE_PMDIS)
		PMDIS3=0xFF;
		PMDIS2=0xFF;
		PMDIS1=0xFF;
		PMDIS0=0xFF;
	#endif  
}

#if (USE_BOOT_UART)
void bootInitUART(void)
{
	unsigned int baudRate;
	
	PMDIS0bits.UART2MD=0;	
	TRISAbits.TRISA0=1;
	ANCON0bits.PCFG0=1;
	TRISAbits.TRISA1=1;
	ANCON0bits.PCFG1=1;
	RCSTA2=0;
	TXSTA2=0;
	BAUDCON2=0;
	ODCON2bits.U2OD=0;
	BAUDCON2bits.RXDTP=1;
	BAUDCON2bits.TXCKP=1;
	BAUDCON2bits.BRG16=1;
	TXSTA2bits.BRGH=1;
	SPBRGH2=(unsigned char)0;
	SPBRG2=(unsigned char)103;		
	TXSTA2bits.TXEN=1;
	RCSTA2bits.CREN=1;		
	RCSTA2bits.SPEN=1;
	RPOR1=6;
	RPINR16=0;
	TRISAbits.TRISA1=0;
}

void bootputrsUART(const rom unsigned char* instring)
{
	while((*instring)!='\0')
	{
		TXREG2=(*instring++);
		while(TXSTA2bits.TRMT==0);
	}
}
#endif

void main(void)
{
	/* Execution Begins Here */
	unsigned long counter;
	unsigned char vectorBlock[64];
	unsigned char code;
	unsigned long timeOut;

	OSCCON = 0x0B;
	ADCON0=((ADCP_CH)<<2)|0x01;
	ADCON1=0x23;
	do {
		ADCON0bits.GO=1;
		while(ADCON0bits.GO);
	} while( ADRESH > 4 && ADRESH < 52 );
	OSCCON = 0x68;

	InitializeSystem();
	bootReset=0;
	flashAddress=0;
	/* Read In the Vector Block! */
	if(!bootFunction)
	{
		ReadFlash(0, 64, (unsigned char*)vectorBlock);
		counter=(((unsigned long)vectorBlock[0x12]<<8)+((unsigned long)vectorBlock[0x10]))<<1;
		/* 
		   Counter Now Points to the beginning of the mainAppInternal function, the entry
		   point for the user's main function. This is because there is a fixed vector at 0x10 
		*/
		ReadFlash(counter, 64, (unsigned char*)vectorBlock);   	
	}
	KEY_TRIS=1;
	if((bootFunction)||(ADRESH <= 4)||((vectorBlock[0]==0xFF)&&(vectorBlock[1]==0xFF)))
	{
	DoBootloader:
		/* Enter the bootloader here! */
		counter=0;
		code=CODE_NONE;
		timeOut=0;
		LED1_TRIS=0;
		LED1=LED1_ON;

		USBDeviceInit();
		USBDeviceAttach();
	
		#if(USE_BOOT_UART)
		bootInitUART();
		bootputrsUART(messageBoot);
		#endif
 
		while((bootReset==0)||(timeOut>0))
		{
			USBDeviceTasks();
			if(!USBHandleBusy(USBGenericOutHandle))
		    {
		    	code=ProcessBootTask(&timeOut);
		    	if(bootReset)
		    	{
		    	
		    	}
		    	else
		    	{
		    		timeOut=BOOT_HOST_TIMEOUT;
		    	}
		    } 
		    else
		    {

		    }

		 	if(timeOut>0)timeOut--;
		 	else
		 	{
		 		code=CODE_NONE;
		 	}	
		 				    
			if(counter>0)
			{
				counter--;
			}
			else
			{
				LED1=!LED1;
				if(LED1==LED1_ON)
				{
					counter=COUNTER_PERIOD_ON;
				}
				else
				{
					switch(code)
		    		{
		    		default:
		    			counter=COUNTER_PERIOD_OFF;
						break;
		    			
		    		case CODE_ONE:
		    			counter=COUNTER_PERIOD_OFF_SHORT;
		    			break;
		    	
		    		case CODE_TWO:
		    			counter=COUNTER_PERIOD_SHORT;
		    			break;
		    		}		
		    	}		    	
			}
		}
		USBSoftDetach();
		_asm
			reset
		_endasm
	}
	/* return to the user, freeing up the stack area used by this routine... */
}

unsigned char ProcessBootTask(unsigned long *timeOutPtr)
{   
	unsigned char* 	PTptr;
	unsigned int 	PTvID;
	unsigned long   PTLong;
	unsigned long   PTLongEnd;
	unsigned char   result;
	unsigned char   mask;
	
	result=CODE_TWO;
	/* Non User code Handlers */    
    switch(OUTPacket.data[0])					//Data arrived, check what kind of command might be in the packet of data.
        { 
	     	default:
	     		result=CODE_NONE;
	     		break;
	     	
	     	case CMD_SET_FLASH_ADDRESS:
	     			if(OUTPacket.data[1]==1)
	     			{
	     			PTptr=(unsigned char*)&flashAddress;
	     			*PTptr++=OUTPacket.data[4];
	           		*PTptr++=OUTPacket.data[5];
	           		*PTptr++=OUTPacket.data[6];
	           		*PTptr++=OUTPacket.data[7];
	        		}   		
	           		PTptr=(unsigned char*)&flashAddress;
	     			INPacket.data[4]=*PTptr++;
					INPacket.data[5]=*PTptr++;
					INPacket.data[6]=*PTptr++;
					INPacket.data[7]=*PTptr++;	
					break;
	           				
	        case CMD_GET_PROGRAM_MEMORY_AT:
	          		/* 
	        			PTvID is the number of bytes and PTLong is the address 
	        			Now Read them...	
	        		*/
	        		ReadFlash((unsigned long)flashAddress, (unsigned int)64, (unsigned char*)&INPacket.data[0]);
	        		flashAddress+=64;
					break;
	         
	         case CMD_PUT_PROGRAM_MEMORY_AT:
	         		if((flashAddress & 0x3F)==0)
	         		{
	         			PTvID=32;
	         			while(PTvID>0)
	         			{
		         			PTvID--;
		         			INPacket.data[PTvID]=OUTPacket.data[4+PTvID];
	         			}
	         			flashAddress+=32;
	         		}
	         		else
	         		{
	         			PTvID=32;
	         			while(PTvID>0)
	         			{
		         			PTvID--;
		         			INPacket.data[PTvID+32]=OUTPacket.data[4+PTvID];
	         			}
	         			flashAddress+=32;
	         			
	         			PTvID=64;
	         			mask=0xFF;
	         			while(PTvID>0)
	         			{
		         			PTvID--;
	         				mask&=INPacket.data[PTvID];
	         			}
	         			/* Skip The Write if they are all 0xFFs */
	         			if(mask!=0xFF)WriteBytesFlash((unsigned long)flashAddress-64, (unsigned int)64, (unsigned char*)&INPacket.data[0]);
	          		}
					break; 
	         
	         case CMD_WRITE_WORD_FLASH:
	        	 	#if(USE_BOOT_WRITE_WORD_FLASH)
			 			 PTptr=(unsigned char*)&PTvID;
	         			 *PTptr++=OUTPacket.data[4];
	         			 *PTptr++=OUTPacket.data[5];
	         			 WriteWordFlash(flashAddress, PTvID);
	        			 flashAddress+=2;
	        		 #endif
	        		 break;
	        		 
			case CMD_ERASE_PROGRAM_MEMORY_AT:
					PTptr=(unsigned char*)&PTLong;
	           		*PTptr++=OUTPacket.data[4];
	           		*PTptr++=OUTPacket.data[5];
	           		*PTptr++=OUTPacket.data[6];
	           		*PTptr++=OUTPacket.data[7];
	           		PTptr=(unsigned char*)&PTLongEnd;
	           		*PTptr++=OUTPacket.data[8];
	           		*PTptr++=OUTPacket.data[9];
	           		*PTptr++=OUTPacket.data[10];
	           		*PTptr++=OUTPacket.data[11];
	           		if((PTLong<PTLongEnd)&&(PTLongEnd<=BOOTLOADER_ADDRESS))PTLong=EraseFlash(PTLong, PTLongEnd);
	           		else PTLong=0;
	           		PTptr=(unsigned char*)&PTLong;
	           		INPacket.data[4]=*PTptr++;
	           		INPacket.data[5]=*PTptr++;
	           		INPacket.data[6]=*PTptr++;
	           		INPacket.data[7]=*PTptr++;
					break;
			
			case CMD_GET_BOOTLOADER_STATUS:
					PTLong=(unsigned long)BOOTLOADER_VERSION;
					PTptr=(unsigned char*)&PTLong;
					INPacket.data[4]=*PTptr++;
					INPacket.data[5]=*PTptr++;
					INPacket.data[6]=*PTptr++;
					INPacket.data[7]=*PTptr++;
					INPacket.data[1]=1;
					result=CODE_ONE;
					break;
					
			case CMD_BOOT_RESET:
					bootReset=1;
					(*timeOutPtr)=BOOT_RESET_TIMEOUT;
					break;
			}
			
	   		USBGenericInHandle = USBGenWrite(USBGEN_EP_NUM,(BYTE*)&INPacket.data[0],USBGEN_EP_SIZE);
       			
	 //Re-arm the OUT endpoint for the next packet:
     //The USBGenRead() function call "arms" the endpoint (and makes it "busy").  If the endpoint is armed, the SIE will 
     //automatically accept data from the host, if the host tries to send a packet of data to the endpoint.  Once a data 
     //packet addressed to this endpoint is received from the host, the endpoint will no longer be busy, and the application
     //can read the data which will be sitting in the buffer.
 	  USBGenericOutHandle = USBGenRead(USBGEN_EP_NUM,(BYTE*)&OUTPacket.data[0], USBGEN_EP_SIZE);   
	  return result;
}
